
#define LOG_INTERVAL_SECONDS 6
#define VRAIL_5              5.000
#define A0_DIV_RATIO        (100.0/47.0)
#define A1_DIV_RATIO        (100.0/47.0)
#define A2_DIV_RATIO        (100.0/47.0)
#define A3_DIV_RATIO        (100.0/47.0)
//#define DS18B20_INPUT        0
//#define COUNTER_INPUT        3
#define COUNTER_AVG_MS       1000
#define LOG_RAM_ENTRIES      6
#define GPS_TIMEOUT         (60*5) // 5 minutes
#define GPS_CHECK_INTERVAL  (60*30) // half an hour
#define SERIAL_DEBUG

// RTCC stuff
#include <Wire.h>
#include "RTClib.h"

RTC_DS3231 rtc;


// SD card stuff
#include <SPI.h>
#include "SdFat.h"

static const uint8_t SD_CS_PIN = SS;
SdFat sd;
SdFile file;
uint8_t sd_file_open;


// GPS stuff
#include <ReceiveOnlyAltSoftSerial.h>
#include "TinyGPS.h"
TinyGPS gps;
#define GPS_RXPIN 8
ReceiveOnlyAltSoftSerial nss;
unsigned long searching_for_gps_since, last_gps_attempt, last_gps_lock, last_last_gps_lock;
signed long gps_lat, gps_lon, last_gps_lat, last_gps_lon;
uint8_t gps_present, gps_active, gps_numsats, last_gps_numsats;


// Other stuff
#include <EEPROM.h>
#ifdef COUNTER_INPUT
#include <MsTimer2.h>
#endif
#ifdef DS18B20_INPUT
#include <OneWire.h>
#endif
#include <avr/wdt.h>
#include <avr/sleep.h>
unsigned long last_log_time;
unsigned char rambuf[(44*LOG_RAM_ENTRIES+7)/8];
#ifdef COUNTER_INPUT
int counterbuf[LOG_RAM_ENTRIES];
#endif
#ifdef DS18B20_INPUT
OneWire ds(DS18B20_INPUT+2);
int16_t tempbuf[LOG_RAM_ENTRIES];
#endif
uint8_t log_ram_filled, pause_logging;
uint32_t last_file_open_unixtime;
float BMP180buf[LOG_RAM_ENTRIES][2];


void FlashForever(uint16_t delayms) {
  while( 1 ) {
    digitalWrite(6, LOW);
    delay(delayms);
    digitalWrite(6, HIGH);
    delay(delayms);
  }
}

#ifdef COUNTER_INPUT
volatile uint16_t LastCount, ThisCount, CounterFreq;
uint8_t LastCounterState;
void CounterReset() {
  LastCount = ThisCount;
  ThisCount = 0;
}
#endif

#ifdef DS18B20_INPUT
byte ds18b20_addr[8];
uint8_t ds18b20_present;
uint8_t detect_ds18b20() {
  byte i;

  ds.reset_search();
  if ( !ds.search(ds18b20_addr)) {
#ifdef SERIAL_DEBUG
    Serial.println(F("Couldn't find DS18B20 sensor."));
#endif
    ds.reset_search();
    return 0;
  } else {
#ifdef SERIAL_DEBUG
    Serial.print(F("Found DS18B20 or similar at address"));
    for( i = 0; i < 8; i++) {
      Serial.print(F(" "));
      Serial.print(ds18b20_addr[i], HEX);
    }
#endif
  
    if ( OneWire::crc8(ds18b20_addr, 7) != ds18b20_addr[7]) {
#ifdef SERIAL_DEBUG
      Serial.println(F("; CRC is not valid!"));
#endif
      return 0;
    }
  
    if ( ds18b20_addr[0] == 0x10 ) {
#ifdef SERIAL_DEBUG
      Serial.println(F("; device is DS18S20 family."));
#endif
    } else if ( ds18b20_addr[0] == 0x28 ) {
#ifdef SERIAL_DEBUG
      Serial.println(F("; device is DS18B20 family."));
#endif
    } else {
#ifdef SERIAL_DEBUG
      Serial.println(F("; device family  not recognized."));
#endif
      return 0;
    }

    ds.reset();
    ds.select(ds18b20_addr);
    ds.write(0x44, 1); // start conversion, with parasite power on at the end
    return 1;
  }
}

int16_t get_ds18b20_temp() {
  byte data[9];
  byte i;

  if( ds.reset() ) {
    int16_t temp;

    ds.select(ds18b20_addr);    
    ds.write(0xBE); // Read Scratchpad

    for ( i = 0; i < 9; i++)
      data[i] = ds.read();
    if( data[8] == OneWire::crc8(data, 8) ) {
      temp = (((uint16_t)data[1])<<8) | data[0];
    } else {
      temp = 0x7FFF;
    }

    ds.reset();
    ds.select(ds18b20_addr);
    ds.write(0x44, 1); // start conversion, with parasite power on at the end
    return temp;
  }
  return 0x7FFF;
}
#endif

void setup() {
  pinMode(2, INPUT_PULLUP);
  pinMode(3, INPUT_PULLUP);
  pinMode(4, INPUT_PULLUP);
  pinMode(5, INPUT_PULLUP);
  pinMode(6, OUTPUT);
  digitalWrite(6, HIGH);
  pinMode(7, OUTPUT);
  pinMode(8, INPUT_PULLUP);
  pinMode(9, INPUT_PULLUP);
  digitalWrite(7, HIGH); // give us a chance to detect the GPS module

#ifndef ESP8266
  while (!Serial); // for Leonardo/Micro/Zero
#endif

  Serial.begin(115200);
#ifdef SERIAL_DEBUG
  Serial.println(F("SILICON CHIP Arduino Datalogger powering up"));
  Serial.flush();
#endif
  if( !rtc.begin() ) {
#ifdef SERIAL_DEBUG
    Serial.println(F("Couldn't find RTC module"));
#endif
    FlashForever(250);
  }

#ifdef DS18B20_INPUT
  ds18b20_present = detect_ds18b20();
#endif

  if( rtc.lostPower() ) {
#ifdef SERIAL_DEBUG
    Serial.println(F("RTC lost power, setting time"));
    Serial.flush();
#endif
    rtc.adjust(DateTime(F(__DATE__), F(__TIME__)));
  } else {
    uint32_t LastSetDateTimeUnix;
    EEPROM.get(0, LastSetDateTimeUnix);
    DateTime CompileDateTime(F(__DATE__), F(__TIME__));
    if( CompileDateTime.unixtime() != LastSetDateTimeUnix ) {
#ifdef SERIAL_DEBUG
      Serial.println(F("RTC time updated"));
      Serial.flush();
#endif
      LastSetDateTimeUnix = CompileDateTime.unixtime();
      EEPROM.put(0, LastSetDateTimeUnix);
      rtc.adjust(CompileDateTime + TimeSpan(20));
    } else {
      DateTime now = rtc.now();
      TimeSpan TimeSinceCompilation = now - CompileDateTime;
      if( TimeSinceCompilation.totalseconds() < 60+20 ) {
#ifdef SERIAL_DEBUG
        Serial.println(F("RTC time advanced to next minute"));
        Serial.flush();
#endif
        now = now + TimeSpan(60 - now.second());
        rtc.adjust(now);
      }
    }
  }

#ifdef COUNTER_INPUT
 #ifdef SERIAL_DEBUG
  Serial.println(F("Calibrating counter"));
 #endif
  MsTimer2::set(1200, CounterReset);
  uint32_t temp = rtc.now().unixtime();
  while( rtc.now().unixtime() == temp )
    ;
  MsTimer2::start();
  ++temp;
  while( rtc.now().unixtime() == temp )
    ;
  MsTimer2::stop();
  CounterFreq = MsTimer2::count * 10UL + (256 - TCNT2) * 10 / (256 - MsTimer2::tcnt2);

  MsTimer2::set(COUNTER_AVG_MS, CounterReset);
  MsTimer2::start();

  #if COUNTER_INPUT == 0
    PCMSK2 |= bit (PCINT18); // D2
  #elif COUNTER_INPUT == 1
    PCMSK2 |= bit (PCINT19); // D3
  #elif COUNTER_INPUT == 2
    PCMSK2 |= bit (PCINT20); // D4
  #elif COUNTER_INPUT == 3
    PCMSK2 |= bit (PCINT21); // D5
  #endif
  PCIFR  |= bit (PCIF2);   // clear any outstanding interrupts
  PCICR  |= bit (PCIE2);   // enable pin change interrupt for D0
  PORTD;
 #ifdef SERIAL_DEBUG
  Serial.print(F("Counter calibration complete ("));
  Serial.print(CounterFreq / 10.0);
  Serial.print(F(")\n"));
  Serial.flush();
 #endif
//  rtc.writeSqwPinMode(DS3231_SquareWave1kHz);
#endif

#ifdef SERIAL_DEBUG
  Serial.println(F("Initialising SD card"));
  Serial.flush();
#endif
  if( !sd.begin(SD_CS_PIN, SPI_FULL_SPEED) ) {
#ifdef SERIAL_DEBUG
    Serial.println(F("Error initialising SD card"));
#endif
    FlashForever(125);
  }

  if( 60 / LOG_INTERVAL_SECONDS * LOG_INTERVAL_SECONDS == 60 ) {
    DateTime now = rtc.now();
    uint8_t mins = now.unixtime()%60;
    last_log_time = now.unixtime() - mins + (mins / LOG_INTERVAL_SECONDS * LOG_INTERVAL_SECONDS);
  } else {
    last_log_time = rtc.now().unixtime();
  }

  bmp085Calibration();
  
#ifdef SERIAL_DEBUG
  Serial.println(F("SILICON CHIP Arduino Datalogger ready"));
#endif
}

ISR(WDT_vect) {
  PORTD;
}

#define WD_RST                  8
#define WD_OFF                  0
#if defined(WDIF)
  #define WD_IRQ                0xC0
  #define WD_RST_IRQ            0xC8
#endif
#define WD_SET(val,...)                                 \
        __asm__ __volatile__(                               \
            "in __tmp_reg__,__SREG__"           "\n\t"      \
            "cli"                               "\n\t"      \
            "wdr"                               "\n\t"      \
            "sts %[wdreg],%[wden]"              "\n\t"      \
            "sts %[wdreg],%[wdval]"             "\n\t"      \
            "out __SREG__,__tmp_reg__"          "\n\t"      \
            :                                               \
            : [wdreg] "M" (&WDTCSR),                        \
              [wden]  "r" ((uint8_t)(0x18)),                \
              [wdval] "r" ((uint8_t)(val|(__VA_ARGS__+0)))  \
            : "r0"                                          \
    )

//ISR(PCINT0_vect) {
//  // pin change interrupt for D8
//}

ISR(PCINT2_vect) {
#ifdef COUNTER_INPUT
  if( digitalRead(COUNTER_INPUT+2) != LastCounterState ) {
    LastCounterState ^= 1;
    if( LastCounterState )
      ++ThisCount;
  }
#endif
}

void SleepMilliseconds(uint16_t sleeptime) {
  uint8_t flag;
//  Serial.print("sleep ");
//  Serial.print(sleeptime);
//  Serial.print("\n");
  Serial.flush();

  if( sleeptime <= 16 )
    flag = WDTO_15MS;
  else if( sleeptime <= 32 )
    flag = WDTO_30MS;
  else if( sleeptime <= 63 )
    flag = WDTO_60MS;
  else if( sleeptime <= 125 )
    flag = WDTO_120MS;
  else if( sleeptime <= 250 )
    flag = WDTO_250MS;
  else if( sleeptime <= 500 )
    flag = WDTO_500MS;
  else if( sleeptime <= 1000 )
    flag = WDTO_1S;
  else if( sleeptime <= 2000 )
    flag = WDTO_2S;
  else if( sleeptime <= 4000 )
    flag = WDTO_4S;
  else
    flag = WDTO_8S;
  WD_SET(WD_IRQ,flag);

  sleep_enable();
#ifdef COUNTER_INPUT
  // need Timer2 to continue running
  set_sleep_mode(SLEEP_MODE_EXT_STANDBY);
#else
  set_sleep_mode(SLEEP_MODE_PWR_DOWN);
#endif

  // pin change interrupt for D0
  PCMSK2 |= bit (PCINT16); // want pin 0
  PCIFR  |= bit (PCIF2);   // clear any outstanding interrupts
#ifndef COUNTER_INPUT
  PCICR  |= bit (PCIE2);   // enable pin change interrupt for D0
#endif
  UCSR0B &= ~bit (RXEN0);  // disable receiver
  UCSR0B &= ~bit (TXEN0);  // disable transmitter
//  PORTD;

  interrupts ();
  sleep_cpu();
  sleep_disable();

#ifndef COUNTER_INPUT
  PCICR  &= ~bit (PCIE2); // disable pin change interrupt for D0
#endif
  UCSR0B |= bit (RXEN0);  // enable receiver
  UCSR0B |= bit (TXEN0);  // enable transmitter

  wdt_disable();
}

void write_RAM_log_entry(uint16_t a0, uint16_t a1, uint16_t a2, uint16_t a3, uint8_t d0, uint8_t d1, uint8_t d2, uint8_t d3) {
  uint8_t offset = log_ram_filled * 11 / 2;
  uint8_t dd = (d0&1)|((d1&1)<<1)|((d2&1)<<2)|((d3&1)<<3);
  rambuf[offset] = a0;
  rambuf[offset+1] = a1;
  rambuf[offset+2] = a2;
  rambuf[offset+3] = a3;
  rambuf[offset+4] = ((a0>>8)&0x03)|((a1>>6)&0x0C)|((a2>>4)&0x30)|((a3>>2)&0xC0);
  if( log_ram_filled&1 )
    rambuf[offset+5] |= dd<<4;
  else
    rambuf[offset+10] = dd;
#ifdef COUNTER_INPUT
  counterbuf[log_ram_filled] = LastCount * 1000000UL / (CounterFreq * (unsigned long)COUNTER_AVG_MS / 10);
#endif
#ifdef DS18B20_INPUT
  if( ds18b20_present )
    tempbuf[log_ram_filled] = get_ds18b20_temp();
#endif

  BMP180buf[log_ram_filled][0] = bmp085GetTemperature(bmp085ReadUT()); // in degrees Celcius
  BMP180buf[log_ram_filled][1] = bmp085GetPressure(bmp085ReadUP()); // in pascals
  ++log_ram_filled;
}
void read_RAM_log_entry(uint16_t& a0, uint16_t& a1, uint16_t& a2, uint16_t& a3, uint8_t& d0, uint8_t& d1, uint8_t& d2, uint8_t& d3) {
  uint8_t offset = log_ram_filled * 11 / 2;
  uint8_t aa, dd;
  aa = rambuf[offset+4];
  a0 = rambuf[offset]   | (((uint16_t)(aa&0x03))<<8);
  a1 = rambuf[offset+1] | (((uint16_t)(aa&0x0C))<<6);
  a2 = rambuf[offset+2] | (((uint16_t)(aa&0x30))<<4);
  a3 = rambuf[offset+3] | (((uint16_t)(aa&0xC0))<<2);
  if( log_ram_filled&1 )
    dd = rambuf[offset+5]>>4;
  else
    dd = rambuf[offset+10];
  d0 = dd&1;
  d1 = (dd>>1)&1;
  d2 = (dd>>2)&1;
  d3 = (dd>>3)&1;
  ++log_ram_filled;
}

static const char LogFileNameTemplate[] PROGMEM = "ArduinoLog_%04d-%02d-%02d_%02d%02d%02d.csv";
static const char TempFormatTemplate[] PROGMEM = "%d.%02d";
#if defined(DS18B20_INPUT) && DS18B20_INPUT == 0
static const char LogEntryTemplate[] PROGMEM = "%02d/%02d/%04d,%02d:%02d:%02d,%d.%02d,%d.%02d,%d.%02d,%d.%02d,%s,%d,%d,%d";
#elif DS18B20_INPUT == 1
static const char LogEntryTemplate[] PROGMEM = "%02d/%02d/%04d,%02d:%02d:%02d,%d.%02d,%d.%02d,%d.%02d,%d.%02d,%d,%s,%d,%d";
#elif DS18B20_INPUT == 2
static const char LogEntryTemplate[] PROGMEM = "%02d/%02d/%04d,%02d:%02d:%02d,%d.%02d,%d.%02d,%d.%02d,%d.%02d,%d,%d,%s,%d";
#elif DS18B20_INPUT == 3
static const char LogEntryTemplate[] PROGMEM = "%02d/%02d/%04d,%02d:%02d:%02d,%d.%02d,%d.%02d,%d.%02d,%d.%02d,%d,%d,%d,%s";
#else
static const char LogEntryTemplate[] PROGMEM = "%02d/%02d/%04d,%02d:%02d:%02d,%d.%02d,%d.%02d,%d.%02d,%d.%02d,%d,%d,%d,%d,%d.%01d,%d.%02d";
#endif
static const char LogFileGPSTemplate[] PROGMEM = ",%ld.%06ld,%ld.%06ld,%d,%lu";
static const char LogFileGPSBlankTemplate[] PROGMEM = ",,,,";

void write_buffered_log_entries() {
  DateTime now(last_log_time);
  uint8_t num_entries = log_ram_filled;
  now = now - TimeSpan( (num_entries-1) * LOG_INTERVAL_SECONDS );
#ifdef COUNTER_INPUT
  char buf[56+38];
#else
  char buf[56+30];
#endif

  if( !sd_file_open ) {
    // work around but in SDFat which fails to put card in low power mode when it isn't in use
    last_file_open_unixtime = now.unixtime();
    sprintf_P(buf, LogFileNameTemplate, now.year(), now.month(), now.day(), now.hour(), now.minute(), now.second());
#ifdef SERIAL_DEBUG
    Serial.print(F("Opening log file "));
    Serial.print(buf);
    Serial.print(F("\n"));
    Serial.flush();
#endif
if( !file.open(buf, O_RDWR | O_CREAT | O_AT_END) )
      FlashForever(125);
    if( !file.println(F("Date,Time,VA0,VA1,VA2,VA3,D0,D1,D2,D3,Lat,Lon,NumSats,SecondsSinceLock")) )
      FlashForever(125);
    Serial.println(buf);
    Serial.flush();
    sd_file_open = 1;
  }

  log_ram_filled = 0;
  do {
    uint16_t a0, a1, a2, a3;
    uint8_t d0, d1, d2, d3;
    float voltage[4];
#ifdef DS18B20_INPUT
    char tempstr[8];
#endif
    read_RAM_log_entry(a0, a1, a2, a3, d0, d1, d2, d3);

    // using 1015 rather than 1023 to compensate for relatively high source impedance
    voltage[0] = a0 * VRAIL_5 * (A0_DIV_RATIO + 1) / 1015;
    voltage[1] = a1 * VRAIL_5 * (A1_DIV_RATIO + 1) / 1015;
    voltage[2] = a2 * VRAIL_5 * (A2_DIV_RATIO + 1) / 1015;
    voltage[3] = a3 * VRAIL_5 * (A3_DIV_RATIO + 1) / 1015;
#ifdef DS18B20_INPUT
    if( tempbuf[log_ram_filled-1] == 0x7FFF )
      tempstr[0] = '\0';
    else
      sprintf_P(tempstr, TempFormatTemplate, tempbuf[log_ram_filled-1] / 16, (int)((tempbuf[log_ram_filled-1] * 25L / 4) % 100));
#endif
    sprintf_P(buf, LogEntryTemplate,
                  now.day(), now.month(), now.year(), now.hour(), now.minute(), now.second(),
                  (int)voltage[0], (int)((int)(voltage[0]*100))%100, (int)voltage[1], (int)((int)(voltage[1]*100))%100,
                  (int)voltage[2], (int)((int)(voltage[2]*100))%100, (int)voltage[3], (int)((int)(voltage[3]*100))%100,
#if defined(COUNTER_INPUT) && COUNTER_INPUT == 0
                  (int)counterbuf[log_ram_filled-1
#elif defined(DS18B20_INPUT) && DS18B20_INPUT == 0
                  tempstr
#else
                  (int)d0
#endif
                  ,
#if COUNTER_INPUT == 1
                  (int)counterbuf[log_ram_filled-1]
#elif DS18B20_INPUT == 1
                  tempstr
#else
                  (int)d1
#endif
                  ,
#if COUNTER_INPUT == 2
                  (int)counterbuf[log_ram_filled-1]
#elif DS18B20_INPUT == 2
                  tempstr
#else
                  (int)d2
#endif
                  ,
#if COUNTER_INPUT == 3
                  (int)counterbuf[log_ram_filled-1]
#elif DS18B20_INPUT == 3
                  tempstr
#else
                  (int)d3
#endif
                  // add any extra logged data here
                 ,(int)BMP180buf[log_ram_filled-1][0]
                 ,(int)((int)(BMP180buf[log_ram_filled-1][0]*10))%10
                 ,(int)(BMP180buf[log_ram_filled-1][1]/100)
                 ,(int)((long)BMP180buf[log_ram_filled-1][1]%100)
    );

    if( gps_present ) {
      if( last_gps_lock ) {
        if( now.unixtime() < last_gps_lock ) {
          if( last_last_gps_lock ) {
            sprintf_P(buf+strlen(buf), LogFileGPSTemplate, last_gps_lat/1000000, labs(last_gps_lat%1000000), last_gps_lon/1000000, labs(last_gps_lon%1000000), (int)last_gps_numsats, now.unixtime()-last_last_gps_lock);
          } else {
            strcpy_P(buf+strlen(buf), LogFileGPSBlankTemplate);
          }
        } else {
          sprintf_P(buf+strlen(buf), LogFileGPSTemplate, gps_lat/1000000, labs(gps_lat%1000000), gps_lon/1000000, labs(gps_lon%1000000), (int)gps_numsats, now.unixtime()-last_gps_lock);
        }
      } else {
        strcpy_P(buf+strlen(buf), LogFileGPSBlankTemplate);
      }
    }
    Serial.println(buf);
    Serial.flush();
    if( !file.println(buf) )
      FlashForever(125);
    now = now + TimeSpan(LOG_INTERVAL_SECONDS);
  } while( log_ram_filled < num_entries );

  log_ram_filled = 0;
}

void update_rtc_time(DateTime& now) {
  int year;
  byte month, day, hour, minute, second;
  unsigned long fix_age;

  gps.crack_datetime(&year, &month, &day, &hour, &minute, &second, 0, &fix_age);
  if( fix_age < 1000 ) {
    DateTime newnow(year, month, day, hour, minute, second);
    TimeSpan diff = newnow - now;

    int32_t diff_secs = diff.totalseconds();
    if( diff_secs >= -108000L && diff_secs <= 108000L /* max 30 hours differece */ ) {
      diff_secs = diff_secs%(30*60);
      if( diff_secs < -15*60 )
        diff_secs += 30*60;
      else if( diff_secs > 15*60 )
        diff_secs -= 30*60;
    }

    if( last_log_time )
      last_log_time += diff_secs;
    if( last_gps_attempt )
      last_gps_attempt += diff_secs;
    if( last_gps_lock )
      last_gps_lock += diff_secs;
    if( last_last_gps_lock )
      last_last_gps_lock += diff_secs;
    now = now + TimeSpan(diff_secs);
    rtc.adjust(now);
#ifdef SERIAL_DEBUG
    Serial.print(F("RTC time updated based on GPS ("));
    Serial.print(diff_secs);
    Serial.println(F(")"));
#endif
  }
}

void loop() {
  DateTime now = rtc.now();
  if( !gps_present ) {
    // if D8 is low and we haven't detected a GPS, start software serial on that pin and wait for a valid data string
    if( !digitalRead(8) && !searching_for_gps_since ) {
#ifdef SERIAL_DEBUG
      Serial.println(F("GPS module might be present, checking..."));
#endif
      searching_for_gps_since = now.unixtime();
      nss.begin(9600);
    } else if( searching_for_gps_since ) {
      // if D8 went low but not valid data string has been received for 10 seconds, stop looking (at least until D8 goes low again)
      if( now.unixtime() - searching_for_gps_since > 10 ) {
#ifdef SERIAL_DEBUG
        Serial.println(F("GPS module not detected (timeout)"));
#endif
        searching_for_gps_since = 0;
        nss.end();
      }
    }
    // if we have started checking for a GPS unit because D8 has gone low, wait until we receieve a valid string and we can then activate the GPS unit
    while( searching_for_gps_since && nss.available() > 6 ) {
      if( nss.read() == '$' && nss.read() == 'G' && nss.read() == 'P' && nss.read() == 'R' && nss.read() == 'M' && nss.read() == 'C' && nss.read() == ',' ) {
        gps_present = 1;
        searching_for_gps_since = 0;
#ifdef SERIAL_DEBUG
        Serial.println(F("GPS module detected"));
#endif
        gps_active = 1;
        gps = TinyGPS();
        last_gps_attempt = now.unixtime();
      }
    }
  }

  // if the GPS unit is active, either immediately after being detected or when the position data has become stale, wait to see if we get a lock
  if( gps_active ) {
    // if a lock is taking too long, switch the GPS unit off and try again later
    if( now.unixtime() - last_gps_attempt > GPS_TIMEOUT ) {
      gps_active = 0;
      digitalWrite(7, LOW); // now switch the module off
#ifdef SERIAL_DEBUG
      Serial.println(F("GPS module time-out before location fix"));
#endif
    } else {
      // if there's data from the GPS unit, decode it and if we have position data, store it and switch the GPS unit off
      while( nss.available() ) {
        char c = nss.read();
        gps.encode(c);
//        Serial.write(c);
      }

      unsigned long gps_fix_age;
      signed long new_lat, new_lon;
      gps.get_position(&new_lat, &new_lon, &gps_fix_age);
      if( gps_fix_age < 1000 && new_lat != TinyGPS::GPS_INVALID_ANGLE && new_lon != TinyGPS::GPS_INVALID_ANGLE && gps.satellites() != 255 ) {
        update_rtc_time(now);

        last_gps_lat = gps_lat;
        last_gps_lon = gps_lon;
        gps_lat = new_lat;
        gps_lon = new_lon;
        last_gps_numsats = gps_numsats;
        gps_numsats = gps.satellites();
        last_last_gps_lock = last_gps_lock;
        last_gps_lock = now.unixtime();

        gps_active = 0;
        digitalWrite(7, LOW); // now switch the module off
#ifdef SERIAL_DEBUG
        Serial.println(F("got GPS fix, powering module down"));
#endif
      }
    }
  } else if( gps_present && now.unixtime() - last_gps_attempt > GPS_CHECK_INTERVAL ) {
    // if it's been a while since the last check for GPS position, switch the unit back on and wait for valid data
    digitalWrite(7, HIGH); // switch the module back on
    gps_active = 1;
    gps = TinyGPS();
    last_gps_attempt = now.unixtime();
#ifdef SERIAL_DEBUG
    Serial.println(F("GPS data stale, powering module up to get new fix"));
#endif
  }

  // check to see whether we should be actively logging or not
  if( digitalRead(9) && !pause_logging ) {
    // if we're logging, now check to see if it's time to create the next log entry
    if( last_log_time + LOG_INTERVAL_SECONDS > now.unixtime() ) {
      // it isn't time to log an entry yet. If the GPS unit is active, we don't go into sleep mode since we might lose data
//      if( nss.busy() ) {
      if( searching_for_gps_since || gps_active ) {
//        delay(10);
      } else {
        // we aren't ready to log an entry and the GPS unit is not switched on, so go into sleep mode for an appropriate time
        uint8_t wait_seconds = (last_log_time + LOG_INTERVAL_SECONDS) - now.unixtime();
        if( wait_seconds > 8 ) {
          SleepMilliseconds(8000);
        } else if( wait_seconds > 4 ) {
          SleepMilliseconds(4000);
        } else if( wait_seconds > 2 ) {
          SleepMilliseconds(2000);
        } else if( wait_seconds > 1 ) {
          SleepMilliseconds(500);
        } else {
          SleepMilliseconds(125);
        }
      }
    } else {
      // it's time to make another log entry so turn LED1 on
      digitalWrite(6, LOW);
      last_log_time += LOG_INTERVAL_SECONDS;

      // read all the input states and save them into one of the RAM buffer slots
      write_RAM_log_entry(analogRead(A0), analogRead(A1), analogRead(A2), analogRead(A3), digitalRead(2), digitalRead(3), digitalRead(4), digitalRead(5));
      // if the RAM buffer is now full, dump the entries to the file on the SD card
      if( log_ram_filled == LOG_RAM_ENTRIES ) {
        // work around but in SDFat which fails to put card in low power mode when it isn't in use
        if( sd_file_open ) {
          char buf[40];
          DateTime past(last_file_open_unixtime);
          sprintf_P(buf, LogFileNameTemplate, past.year(), past.month(), past.day(), past.hour(), past.minute(), past.second());
          if( !file.open(buf, O_RDWR | O_CREAT | O_AT_END) )
            FlashForever(125);
        }
        write_buffered_log_entries();
        // work around but in SDFat which fails to put card in low power mode when it isn't in use
        file.close();
        if( !sd.begin(SD_CS_PIN, SPI_FULL_SPEED) ) {
#ifdef SERIAL_DEBUG
          Serial.println(F("Error initialising SD card"));
#endif
          FlashForever(125);
        }
      }

      // finished logging, so turn LED1 back off
      digitalWrite(6, HIGH);
    }
  } else {
    // we're not actively logging, because either D9 is pulled low or logging has been paused via the serial console so close the log file if it's open
    last_log_time = now.unixtime();
    if( sd_file_open ) {
      digitalWrite(6, LOW);
      write_buffered_log_entries();
#ifdef SERIAL_DEBUG
      Serial.println(F("Closing log file.\n"));
#endif
      file.println(F("Closing log."));
      file.close();
      // work around but in SDFat which fails to put card in low power mode when it isn't in use
      if( !sd.begin(SD_CS_PIN, SPI_FULL_SPEED) ) {
#ifdef SERIAL_DEBUG
        Serial.println(F("Error initialising SD card"));
#endif
        FlashForever(125);
      }
      sd_file_open = 0;
      delay(250);
      digitalWrite(6, HIGH);
    } else {
      // not logging, log file is closed, so go into sleep mode if the GPS module is not active
//      if( nss.busy() ) {
      if( searching_for_gps_since || gps_active ) {
//        delay(10);
      } else {
        SleepMilliseconds(250);
      }
    }
  }

  // we've finished all the logging related tasks, so now check to see if we've received a command via the serial console
  // valid commands are: stop, cont, list, dump
  // "stop" pauses logging
  // "cont" resumes logging
  // "list" writes a list of the log files that are on the microSD card over the serial console
  // "dump <filename>" writes the contents of the specified file to the serial console
  if( Serial.available() > 4 ) {
    char c = Serial.read();
    Serial.print(c);
    Serial.println("'\n");
    if( c == 'c' ) {
      if( Serial.read() == 'o' && Serial.read() == 'n' && Serial.read() == 't' && Serial.read() == '\n' ) {
        if( pause_logging )
          Serial.println(F("Logging resumed"));
        pause_logging = 0;
      } else {
        Serial.println(F("?"));
      }
    } else if( c == 's' ) {
      if( Serial.read() == 't' && Serial.read() == 'o' && Serial.read() == 'p' && Serial.read() == '\n' ) {
        if( !pause_logging )
          Serial.println(F("Logging paused"));
        pause_logging = 1;
      } else {
        Serial.println(F("?"));
      }
    } else if( c == 'l' ) {
      if( Serial.read() == 'i' && Serial.read() == 's' && Serial.read() == 't' && Serial.read() == '\n' ) {
        if( !sd_file_open ) {
          SdFile dirFile;
          if( !dirFile.open("/", O_READ) ) {
            Serial.println(F("Error reading from microSD card"));
          } else {
            Serial.println(F("List of log files on microSD card:"));
            while( file.openNext(&dirFile, O_READ) ) {
              // Skip directories and hidden files.
              if( !file.isSubDir() && !file.isHidden() ) {
                char buf[14];
                if( file.getName(buf, 13) && buf[0] == 'A' && buf[1] == 'r' && buf[2] == 'd' && buf[3] == 'u' && buf[4] == 'i' && buf[5] == 'n' && buf[6] == 'o' &&
                                             buf[7] == 'L' && buf[8] == 'o' && buf[9] == 'g' && buf[10] == '_' ) {
                  file.printName(&Serial);
                  Serial.println();
                }
              }
              file.close();
            }
          }
          dirFile.close();
          Serial.println(F("---"));
          Serial.flush();
        } else {
          Serial.println(F("Can't list files while logging"));
        }
      } else {
        Serial.println(F("?"));
      }
    } else if( c == 'd' ) {
      if( Serial.read() == 'u' && Serial.read() == 'm' && Serial.read() == 'p' && Serial.read() == ' ' ) {
        if( !sd_file_open ) {
          char buf[64];
          uint8_t index = 0;
          delay(1);
          while( index < 64 && Serial.available() ) {
            delay(1);
            buf[index] = Serial.read();
            if( buf[index] == '\n' ) {
              buf[index] = '\0';
              index = 255;
              break;
            }
            ++index;
          }
          if( index != 255 || !file.open(buf, O_READ) ) {
            Serial.println(F("File not found"));
          } else {
            int c;
            while( (c = file.read()) > 0 )
              Serial.write(c);
            if( c != '\n' )
              Serial.println();
            file.close();
            Serial.flush();
          }
        } else {
          Serial.println(F("Can't dump files while logging"));
        }
      } else {
        Serial.println(F("?"));
      }
    } else {
      Serial.println(F("?"));
    }
  }
}

#include <Wire.h>

#define BMP085_ADDRESS 0x77  // I2C address of BMP085

const unsigned char OSS = 0;  // Oversampling Setting

// Calibration values
int ac1;
int ac2;
int ac3;
unsigned int ac4;
unsigned int ac5;
unsigned int ac6;
int b1;
int b2;
int mb;
int mc;
int md;

// b5 is calculated in bmp085GetTemperature(...), this variable is also used in bmp085GetPressure(...)
// so ...Temperature(...) must be called before ...Pressure(...).
long b5; 

// Stores all of the bmp085's calibration values into global variables
// Calibration values are required to calculate temp and pressure
// This function should be called at the beginning of the program
void bmp085Calibration()
{
  ac1 = bmp085ReadInt(0xAA);
  ac2 = bmp085ReadInt(0xAC);
  ac3 = bmp085ReadInt(0xAE);
  ac4 = bmp085ReadInt(0xB0);
  ac5 = bmp085ReadInt(0xB2);
  ac6 = bmp085ReadInt(0xB4);
  b1 = bmp085ReadInt(0xB6);
  b2 = bmp085ReadInt(0xB8);
  mb = bmp085ReadInt(0xBA);
  mc = bmp085ReadInt(0xBC);
  md = bmp085ReadInt(0xBE);
}

// Calculate temperature in deg C
float bmp085GetTemperature(unsigned int ut){
  long x1, x2;

  x1 = (((long)ut - (long)ac6)*(long)ac5) >> 15;
  x2 = ((long)mc << 11)/(x1 + md);
  b5 = x1 + x2;

  float temp = ((b5 + 8)>>4);
  temp = temp /10;

  return temp;
}

// Calculate pressure given up
// calibration values must be known
// b5 is also required so bmp085GetTemperature(...) must be called first.
// Value returned will be pressure in units of Pa.
long bmp085GetPressure(unsigned long up){
  long x1, x2, x3, b3, b6, p;
  unsigned long b4, b7;

  b6 = b5 - 4000;
  // Calculate B3
  x1 = (b2 * (b6 * b6)>>12)>>11;
  x2 = (ac2 * b6)>>11;
  x3 = x1 + x2;
  b3 = (((((long)ac1)*4 + x3)<<OSS) + 2)>>2;

  // Calculate B4
  x1 = (ac3 * b6)>>13;
  x2 = (b1 * ((b6 * b6)>>12))>>16;
  x3 = ((x1 + x2) + 2)>>2;
  b4 = (ac4 * (unsigned long)(x3 + 32768))>>15;

  b7 = ((unsigned long)(up - b3) * (50000>>OSS));
  if (b7 < 0x80000000)
    p = (b7<<1)/b4;
  else
    p = (b7/b4)<<1;

  x1 = (p>>8) * (p>>8);
  x1 = (x1 * 3038)>>16;
  x2 = (-7357 * p)>>16;
  p += (x1 + x2 + 3791)>>4;

  long temp = p;
  return temp;
}

// Read 1 byte from the BMP085 at 'address'
char bmp085Read(unsigned char address)
{
  unsigned char data;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 1);
  while(!Wire.available())
    ;

  return Wire.read();
}

// Read 2 bytes from the BMP085
// First byte will be from 'address'
// Second byte will be from 'address'+1
int bmp085ReadInt(unsigned char address)
{
  unsigned char msb, lsb;

  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(address);
  Wire.endTransmission();

  Wire.requestFrom(BMP085_ADDRESS, 2);
  while(Wire.available()<2)
    ;
  msb = Wire.read();
  lsb = Wire.read();

  return (int) msb<<8 | lsb;
}

// Read the uncompensated temperature value
unsigned int bmp085ReadUT(){
  unsigned int ut;

  // Write 0x2E into Register 0xF4
  // This requests a temperature reading
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x2E);
  Wire.endTransmission();

  // Wait at least 4.5ms
  delay(5);

  // Read two bytes from registers 0xF6 and 0xF7
  ut = bmp085ReadInt(0xF6);
  return ut;
}

// Read the uncompensated pressure value
unsigned long bmp085ReadUP(){

  unsigned char msb, lsb, xlsb;
  unsigned long up = 0;

  // Write 0x34+(OSS<<6) into register 0xF4
  // Request a pressure reading w/ oversampling setting
  Wire.beginTransmission(BMP085_ADDRESS);
  Wire.write(0xF4);
  Wire.write(0x34 + (OSS<<6));
  Wire.endTransmission();

  // Wait for conversion, delay time dependent on OSS
  delay(2 + (3<<OSS));

  // Read register 0xF6 (MSB), 0xF7 (LSB), and 0xF8 (XLSB)
  msb = bmp085Read(0xF6);
  lsb = bmp085Read(0xF7);
  xlsb = bmp085Read(0xF8);

  up = (((unsigned long) msb << 16) | ((unsigned long) lsb << 8) | (unsigned long) xlsb) >> (8-OSS);

  return up;
}

void writeRegister(int deviceAddress, byte address, byte val) {
  Wire.beginTransmission(deviceAddress); // start transmission to device 
  Wire.write(address);       // send register address
  Wire.write(val);         // send value to write
  Wire.endTransmission();     // end transmission
}

int readRegister(int deviceAddress, byte address){

  int v;
  Wire.beginTransmission(deviceAddress);
  Wire.write(address); // register to read
  Wire.endTransmission();

  Wire.requestFrom(deviceAddress, 1); // read a byte

  while(!Wire.available()) {
    // waiting
  }

  v = Wire.read();
  return v;
}

float calcAltitude(float pressure){

  float A = pressure/101325;
  float B = 1/5.25588;
  float C = pow(A,B);
  C = 1 - C;
  C = C /0.0000225577;

  return C;
}

